#!/bin/sh
###############################################################################
#
#                     Update systems root filesystem
#

# application tag for logging
APP_TAG=SystemUpdater

# list of files to backup
BACKUP_FILE_LIST=$UPD_ROOT/backup.list
# where to put backed up filesystem
BACKUP_DIR=$BACKUP_DIR
# location of backed up root filesystem
BACKUP_FS_TAR=$BACKUP_DIR/rootfs.tar


	
##
#	Helper functions
#

log()
{
#   $1 - Log messge
    echo -e "$(date) - $1" | tee -a $UPD_LOG
    #logger -t $APP_TAG $1
}
error()
{
    log "$2"
    echo -e "$(date) - $1" | tee -a $ERR_LOG
    exit $1
}



##
#	Update from the initial rootfs layout
#
UpdateFromVersion1()
{
    log "Starting update from version 1"

    ##
    #	Update filesystem
    #
    log "Update filesystem"
    mkdir -p /rootfs/oldfs

    gzip -cd $UPD_ROOT/rootfs.tar.gz | tar -x -C /rootfs || log "Failed to update filesystem!"

    cp -dpf $UPD_ROOT/mountall.sh /etc/init.d/
    chmod u+rwx /etc/init.d/mountall.sh

    cp /root/gw5Setup.bin /rootfs/root/
}

##
#	Update from the buildroot rootfs layout
#
UpdateFromVersion2()
{
    log "Starting update from version 2"
    

    kernel_updated=0
    product_file=/usr/local/etc/telem/product
    NANDWRITE=/usr/sbin/nandwrite
    uname_m=$(uname -m)
    # With "Device Tree Support"
    dt_model=$([ -f /proc/device-tree/model ] && cat /proc/device-tree/model)
    cpu_hardware=$(echo $(grep "Hardware" /proc/cpuinfo) $dt_model)

    log "$cpu_hardware"

    cpu_type_in_file=$(sed 's/,/\\|/g' "$UPD_ROOT/Board" "$UPD_ROOT/Boards" 2>/dev/null | tail -n1)
    
    if [ -f "$UPD_ROOT/withDT" ]; then
        [ -f /proc/device-tree/model ] || error 1 "Firmware requires device tree, stopping update."
    fi
    
    test -e $product_file && {
        product_name_sys=$(cat $product_file)
        product_name_upd=$(cat $UPD_ROOT/product)
        if [ _"$product_name_sys" != _"$product_name_upd" ]; then
            error 1 "Product name mismatch ('$product_name_sys' != '$product_name_upd'), stopping update!"
        fi
    }

    echo "$cpu_hardware" | grep "$cpu_type_in_file" || {
        error 1 "CPU type mismatch ('$cpu_hardware' != '$cpu_type_in_file'), stopping update."
    }
   
    # parse cpu
    cpu_board=$(echo $cpu_hardware | grep -o -- 'PXA270\|VMX25\|VMX51\|VMX53\|M502-SOM\|M501\|TX6UL\|TX6DL\|TX6QP') || cpu_board=unknown
    
    # default settings
    FLASH_ERASE=/usr/sbin/flash_erase
    kernel_image=$UPD_ROOT/zImage
    nand_flash=true
    # cpu specific
    case $cpu_board in
        PXA270) # Voipac PXA270 Module
            KERNEL_MTD=/dev/mtd1
            if [ -f $FLASH_ERASE ]; then
                erase_cmd="$FLASH_ERASE $KERNEL_MTD 0 0"
            else
                FLASH_ERASE=/usr/sbin/flash_eraseall
                erase_cmd="$FLASH_ERASE $KERNEL_MTD"
            fi
        ;;
        TX6UL|TX6DL|TX6QP) # Ka-Ro electronics TX6UL/DL/QP Board
            kernel_image=$UPD_ROOT/uImage
            nand_flash=false
        ;;
        VMX25|VMX53) # Voipac Technologies VMX25/VMX53 Board
            KERNEL_MTD=/dev/mtd2
            erase_cmd="$FLASH_ERASE $KERNEL_MTD 0 0"
        ;;
        VMX51) # Voipac Technologies VMX51 Board
            KERNEL_MTD=/dev/mtd3
            erase_cmd="$FLASH_ERASE $KERNEL_MTD 0 0"
        ;;
        M502-SOM) # Artila Electronics M502-SOM Board
            kernel_image=$UPD_ROOT/uImage
            KERNEL_MTD=/dev/mtd3
            erase_cmd="$FLASH_ERASE $KERNEL_MTD 0 0"
        ;;
        *) # default
            erase_cmd=false
        ;;
    esac

    if [ -e $kernel_image ]; then
        log "Starting kernel update"
        if $nand_flash; then
            # Check for necessary tools
            test -e "$FLASH_ERASE" || error 1 "Utility '$FLASH_ERASE' not available"
            test -e "$NANDWRITE"   || error 1 "Utility '$NANDWRITE' not available"
        
            # erase flash
            eval $erase_cmd || error 1 "Failed to erase nand flash."
            log "Old kernel erased."
            log "Writing kernel '$kernel_image' to flash '$KERNEL_MTD'"
            # write new kernel to flash
            nandwrite -p $KERNEL_MTD $kernel_image || error 1 "Failed to write new kernel to nand flash."
            log "New kernel written to nand flash."
        else
            mmc=/dev/mmcblk0p1
            # find real mmc kernel partition, because it may be SD card
            for dir in /sys/class/block/mmcblk*p1/; do
                dir=${dir%*/}
                grep -Fq -- MMC "$dir/../device/type" && { mmc="/dev/${dir##*/}";break; }
            done
            mkdir -p /tmp/mmc/kernel/ 
            mount "$mmc" /tmp/mmc/kernel/ || error 1 "Failed to mount mmc kernel partition '$mmc'"
            log "Copying kernel '$kernel_image' to '$mmc'"
            cp $kernel_image /tmp/mmc/kernel/uImage || error 1 "Failed to update kernel"
            sync
            umount /tmp/mmc/kernel/
            log "New kernel written to mmc"
        fi
        kernel_updated=1
        # remove old kernel modules
        rm -rf /lib/modules/*
    fi
    
    
    ##
    #	Update filesystem
    #
    log "Update filesystem"
    #mkdir -p /rootfs/oldfs

    _current=/etc/current

    mkdir -p $_current
    chmod u=rw,o= $_current

    echo "Backing up groups and users"
    cp -pf /etc/group $_current/
    cp -pf /etc/passwd $_current/
    cp -pf /etc/shadow $_current/
    
    echo "Removing old WWW directory"
    ! rm -rf /var/www/*
    
    ! rm -rf /usr/local/bin/post_update/*
    
    local rootfs=$(ls -1 -- $UPD_ROOT/rootfs.tar* | tail -n1)
    case "${rootfs##*.}" in 
        bz2)  unpack_cmd="bzcat $rootfs    | tar -x -C /" ;;
        gz)   unpack_cmd="gzip -cd $rootfs | tar -x -C /" ;;
        lzma) unpack_cmd="lzcat $rootfs    | tar -x -C /" ;;
        tar)  unpack_cmd="tar -x -C / $rootfs"            ;;
        xz)   unpack_cmd="xzcat $rootfs    | tar -x -C /" ;;
#         *)    unpack_cmd=false                            ;;
    esac

    eval $unpack_cmd || error 1 "Failed to update filesystem!"

    test -e $_current && {
        echo "Restoring groups and users"
        mv $_current/* /etc/
        rm -rf $_current
    }

    # If we reach here, then files were extracted and linux configuration is using defaults
    # We need to reload configuration.
    if [ -e /usr/local/etc/telem/setup.tar.xz ]; then 
        # if tar.xz config file exists, reload it
        log "Reload setup.tar.xz"
        mv /usr/local/etc/telem/setup.tar.xz /usr/local/etc/telem/setup.new.tar.xz
    else
        # if tar.xz config doesn't exist and binary exists, reload the binary version
        test -e /etc/init.d/S12telem-config-bin && {
            test -e /usr/local/etc/telem/gwSetup.bin && {
                log "Reload gwSetup.bin"
                mv /usr/local/etc/telem/gwSetup.bin /usr/local/etc/telem/gwSetup.bin.new
            }
        }
    fi

    chmod +x /usr/local/bin/telem/check_permissions

    /usr/local/bin/telem/check_permissions /

    /usr/local/bin/telem/post_update.sh
}


##
#			The Script
#

##
#	Clean unneccessary files/directories
#
rm -R /root/upd/backup
rm -R /oldfs/root/upd
# remove old logs
rm -R /oldfs/root/log
# remove gzipped logs
rm /root/log/archive/*.gz


[ -e /usr/bin/tar ] && rm /bin/tar
##
# Select update to perform
#


if [ -e /linuxrc ]; then
	# gateway has update from buildroot
	UpdateFromVersion2
else
	UpdateFromVersion1
fi


exit 0
