#!/bin/sh
###############################################################################
#
#                     Update systems root filesystem
#

##
#	Helper functions
#

log() {
    # $1 - Log messge
    echo -e "$(date) - $1" | tee -a "${UPD_LOG}"
}
error() {
    log "$2"
    echo -e "$(date) - $1" | tee -a "${ERR_LOG}"
    exit $1
}

reloadconf () {
    local SETUP_FILENAME_NEW='/usr/local/etc/telem/setup.new.tar.xz'
    local SETUP_FILENAME='/usr/local/etc/telem/setup.tar.xz'

    if [ -e "${SETUP_FILENAME}" ]; then
        # if tar.xz config file exists, reload it
        log "Reload setup"
        mv "${SETUP_FILENAME}" "${SETUP_FILENAME_NEW}"
    else
        log "Config not found for reload"
    fi
}

##
#	Update from the buildroot rootfs layout
#
UpdateFromVersion2()
{
    log "Starting update from version 2"

    kernel_updated=0
    product_file='/usr/local/etc/telem/product'
    NANDWRITE='/usr/sbin/nandwrite'
    uname_m="$(uname -m)"
    # With "Device Tree Support"
    dt_model="$([ -f '/proc/device-tree/model' ] && cat '/proc/device-tree/model')"
    cpu_hardware="$(echo "$(grep "Hardware" /proc/cpuinfo)" "${dt_model}")"

    log "${cpu_hardware}"

    cpu_type_in_file="$(sed 's/,/\\|/g' "${UPD_ROOT}/Board" "${UPD_ROOT}/Boards" 2>/dev/null | tail -n1)"

    if [ -f "${UPD_ROOT}/withDT" ]; then
        [ -f '/proc/device-tree/model' ] || error 1 "Firmware requires device tree, stopping update."
    fi

    if test -e "${product_file}"; then
        product_name_sys="$(cat "${product_file}")"
        product_name_upd="$(cat "${UPD_ROOT}/product")"
        if [ "${product_name_sys}" != "${product_name_upd}" ]; then
            error 1 "Product name mismatch ('${product_name_sys}' != '${product_name_upd}'), stopping update!"
        fi
    fi

    echo "${cpu_hardware}" | grep "${cpu_type_in_file}" || {
        error 1 "CPU type mismatch ('${cpu_hardware}' != '${cpu_type_in_file}'), stopping update."
    }

    # test with gwconf
    if [ -d "${UPD_ROOT}/gwconf-files" ]; then
        log "Start gwconf test"
        "${UPD_ROOT}/gwconf-files/gwconf-fwtest" 2>>"${ERR_LOG}" \
        || error 1 "gwconf test failed, stopping update."
    else
        log "No gwconf"
    fi

    # parse cpu
    cpu_board=$(echo $cpu_hardware | grep -o -- 'PXA270\|VMX25\|VMX51\|VMX53\|M502-SOM\|M501\|TX6UL\|TX6ULL\|TX6DL\|TX6QP') || cpu_board=unknown

    # default settings
    FLASH_ERASE='/usr/sbin/flash_erase'
    kernel_image="${UPD_ROOT}/zImage"
    bool_nand_flash=true
    # cpu specific
    case "${cpu_board}" in
        TX6UL|TX6ULL|TX6DL|TX6QP) # Ka-Ro electronics TX6UL/ULL/DL/QP Board
            kernel_image="${UPD_ROOT}/uImage"
            bool_nand_flash=false
        ;;
        VMX25|VMX53) # Voipac Technologies VMX25/VMX53 Board
            KERNEL_MTD='/dev/mtd2'
            erase_cmd="${FLASH_ERASE} ${KERNEL_MTD} 0 0"
        ;;
        *) # default
            erase_cmd=false
        ;;
    esac

    if [ -e "${kernel_image}" ]; then
        log "Starting kernel update"
        if ${bool_nand_flash}; then
            # Check for necessary tools
            test -e "${FLASH_ERASE}" || error 1 "Utility '${FLASH_ERASE}' not available"
            test -e "${NANDWRITE}"   || error 1 "Utility '${NANDWRITE}' not available"

            # erase flash
            eval "${erase_cmd}" 2>>"${ERR_LOG}" || error 1 "Failed to erase nand flash."
            log "Old kernel erased."
            log "Writing kernel '${kernel_image}' to flash '${KERNEL_MTD}'"
            # write new kernel to flash
            nandwrite -p "${KERNEL_MTD}" "${kernel_image}" 2>>"${ERR_LOG}" || error 1 "Failed to write new kernel to nand flash."
            log "New kernel written to nand flash."
        else
            mmc='/dev/mmcblk0p1'
            # find real mmc kernel partition, because it may be SD card
            for dir in '/sys/class/block'/mmcblk*p1/; do
                dir="${dir%*/}"
                grep -Fq -- 'MMC' "${dir}/../device/type" && { mmc="/dev/${dir##*/}";break; }
            done
            mkdir -p '/tmp/mmc/kernel/'
            mount "${mmc}" '/tmp/mmc/kernel/' 2>>"${ERR_LOG}" || error 1 "Failed to mount mmc kernel partition '${mmc}'"
            log "Copying kernel '${kernel_image}' to '${mmc}'"
            cp "${kernel_image}" '/tmp/mmc/kernel/uImage' 2>>"${ERR_LOG}" || error 1 "Failed to update kernel"
            sync
            umount '/tmp/mmc/kernel/'
            log "New kernel written to mmc"
        fi
        kernel_updated=1
        # remove old kernel modules
        rm -rf '/lib/modules'/*
    fi

    log "Update filesystem"
    #mkdir -p /rootfs/oldfs

    _current='/etc/current'

    mkdir -p "${_current}"
    chmod u=rw,o= "${_current}"

    echo "Backing up groups and users"
    cp -pf '/etc/group'  "${_current}/"
    cp -pf '/etc/passwd' "${_current}/"
    cp -pf '/etc/shadow' "${_current}/"

    echo "Removing old WWW directory"
    ! rm -rf '/var/www'/*

    ! rm -rf '/usr/local/bin/post_update'/*
    
    # remove /dev/fd, new buildroot wants to create symlink
    [ -d '/dev/fd' ] && rm -rf '/dev/fd'

    local rootfs="$(ls -1 -- "${UPD_ROOT}"/rootfs.tar* | tail -n1)"
    case "${rootfs##*.}" in
        bz2)  unpack_cmd="bzcat    ${rootfs} | tar -x -C /" ;;
        gz)   unpack_cmd="gzip -cd ${rootfs} | tar -x -C /" ;;
        lzma) unpack_cmd="lzcat    ${rootfs} | tar -x -C /" ;;
        tar)  unpack_cmd="tar -x -C / ${rootfs}"            ;;
        xz)   unpack_cmd="xzcat    ${rootfs} | tar -x -C /" ;;
#         *)    unpack_cmd=false                            ;;
    esac

    eval "${unpack_cmd}" 2>>"${ERR_LOG}" || error 1 "Failed to update filesystem!"

    if test -e "${_current}"; then
        echo "Restoring groups and users"
        mv "${_current}"/* /etc/
        rm -rf "${_current}"
    fi

    # If we reach here, then files were extracted and linux configuration is using defaults
    # We need to reload configuration.
    reloadconf

    chmod +x '/usr/local/bin/telem/check_permissions'

    /usr/local/bin/telem/check_permissions /

    /usr/local/bin/telem/post_update.sh
}


##
#			The Script
#

##
#	Clean unneccessary files/directories
#
rm -R /root/upd/backup
rm -R /oldfs/root/upd
# remove old logs
rm -R /oldfs/root/log
# remove gzipped logs
rm /root/log/archive/*.gz


[ -e /usr/bin/tar ] && rm /bin/tar

UpdateFromVersion2

exit 0
